<?php

namespace App\Core\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Core\Models\CompanyBranch;
use Illuminate\Support\Facades\Validator;

class BranchController extends Controller
{
    /**
     * GET /api/branches?company_id=ID
     * Fetch branches by company
     */
    public function index(Request $request)
    {
        $companyId = $request->query('company_id');

        if (!$companyId) {
            return response()->json([], 200);
        }

        $branches = CompanyBranch::where('company_id', $companyId)
            ->orderBy('id', 'desc')
            ->get();

        return response()->json($branches, 200);
    }

    /**
     * POST /api/branches
     * Create new branch
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'company_id'      => 'required|exists:companies,id',
            'name'            => 'required|string|max:255',
            'code'            => 'nullable|string|max:100',
            'type'            => 'nullable|string|max:50',
            'country'         => 'nullable|string|max:100',
            'state'           => 'nullable|string|max:100',
            'city'            => 'nullable|string|max:100',
            'address'         => 'nullable|string',
            'latitude'        => 'nullable|string',
            'longitude'       => 'nullable|string',
            'is_active'       => 'boolean',
            'is_sync_allowed' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $branch = CompanyBranch::create([
            'company_id'      => $request->company_id,
            'name'            => $request->name,
            'code'            => $request->code,
            'type'            => $request->type,
            'country'         => $request->country,
            'state'           => $request->state,
            'city'            => $request->city,
            'address'         => $request->address,
            'latitude'        => $request->latitude,
            'longitude'       => $request->longitude,
            'is_active'       => $request->is_active ?? true,
            'is_sync_allowed' => $request->is_sync_allowed ?? true,
        ]);

        return response()->json($branch, 201);
    }

    /**
     * PUT /api/branches/{id}
     * Update branch
     */
    public function update(Request $request, $id)
    {
        $branch = CompanyBranch::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name'            => 'required|string|max:255',
            'code'            => 'nullable|string|max:100',
            'type'            => 'nullable|string|max:50',
            'country'         => 'nullable|string|max:100',
            'state'           => 'nullable|string|max:100',
            'city'            => 'nullable|string|max:100',
            'address'         => 'nullable|string',
            'latitude'        => 'nullable|string',
            'longitude'       => 'nullable|string',
            'is_active'       => 'boolean',
            'is_sync_allowed' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $branch->update($request->only([
            'name',
            'code',
            'type',
            'country',
            'state',
            'city',
            'address',
            'latitude',
            'longitude',
            'is_active',
            'is_sync_allowed',
        ]));

        return response()->json($branch, 200);
    }

    /**
     * DELETE /api/branches/{id}
     * Delete branch
     */
    public function destroy($id)
    {
        $branch = CompanyBranch::findOrFail($id);
        $branch->delete();

        return response()->json([
            'message' => 'Branch deleted successfully'
        ], 200);
    }

    /**
     * PATCH /api/branches/{id}/toggle-active
     * Toggle active status
     */
    public function toggleActive($id)
    {
        $branch = CompanyBranch::findOrFail($id);

        $branch->is_active = !$branch->is_active;
        $branch->save();

        return response()->json([
            'id'        => $branch->id,
            'is_active' => $branch->is_active
        ], 200);
    }
}
