<?php
namespace App\Core\Http\Controllers;

use App\Core\Models\Company;
use App\Core\Models\Branch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\ModuleService;
use App\Http\Controllers\Controller;

class CoreController extends Controller
{
    // --- GET ALL COMPANIES ---
public function index(): JsonResponse
{
    $companies = Company::all();

    return response()->json([
        'status' => true,
        'message' => 'Companies retrieved successfully',
        'data' => $companies,
    ]);
}


    // --- GET SINGLE COMPANY ---
    public function show(int $id): JsonResponse
    {
        $company = Company::find($id);

        if (!$company) {
            return response()->json([
                'status' => false,
                'message' => 'Company not found',
            ], 404);
        }

        return response()->json([
            'status' => true,
            'message' => 'Company retrieved successfully',
            'data' => $company,
        ]);
    }

    // --- CREATE COMPANY ---
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'legal_name' => 'nullable|string|max:255',
            'registration_no' => 'nullable|string|max:255',
            'tax_id' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'city' => 'nullable|string|max:100',
            'address' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'logo' => 'nullable|string|max:255',
            'theme' => 'nullable|string|max:50',
            'timezone' => 'nullable|string|max:100',
            'date_format' => 'nullable|string|max:50',
            'is_active' => 'nullable|boolean',
        ]);

        DB::beginTransaction();

        try {
            $company = Company::create($validated);
            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Company created successfully',
                'data' => $company
            ], 201);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to create company',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // --- UPDATE COMPANY ---
    public function update(Request $request, int $id): JsonResponse
    {
        $company = Company::find($id);

        if (!$company) {
            return response()->json([
                'status' => false,
                'message' => 'Company not found',
            ], 404);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'legal_name' => 'nullable|string|max:255',
            'registration_no' => 'nullable|string|max:255',
            'tax_id' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'city' => 'nullable|string|max:100',
            'address' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'logo' => 'nullable|string|max:255',
            'theme' => 'nullable|string|max:50',
            'timezone' => 'nullable|string|max:100',
            'date_format' => 'nullable|string|max:50',
            'is_active' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            $company->update($validated);
            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Company updated successfully',
                'data' => $company
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to update company',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // --- DELETE COMPANY ---
    public function destroy(int $id)
    {
        $company = Company::find($id);

        if (!$company) {
            return response()->json([
                'status' => false,
                'message' => 'Company not found',
            ], 404);
        }

        try {
            $company->delete();
            return response()->json([
                'status' => true,
                'message' => 'Company deleted successfully',
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete company',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // --- TOGGLE ACTIVE STATUS ---
    public function toggleActive(int $id): JsonResponse
    {
        $company = Company::find($id);

        if (!$company) {
            return response()->json([
                'status' => false,
                'message' => 'Company not found',
            ], 404);
        }

        try {
            $company->is_active = !$company->is_active;
            $company->save();

            return response()->json([
                'status' => true,
                'message' => 'Company status updated successfully',
                'data' => $company
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to toggle company status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // --- Test / Status Functions ---
    public function testCore(): JsonResponse
    {
        return response()->json([
            'status' => true,
            'message' => 'Core layer is ready!',
            'data' => ['core' => 'working']
        ]);
    }  

    public function status(): JsonResponse
    {
        return response()->json([
            'status' => true,
            'message' => 'Core layer is ready!',
            'data' => ['core' => 'working']
        ]);
    }

    public function modules(ModuleService $moduleService): JsonResponse
    {
        return response()->json($moduleService->getAllModules());
    }
}
