<?php

namespace App\Modules\AccessControl\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Modules\AccessControl\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class AccessControlController extends Controller
{
    /**
     * Check module status
     */
    public function status()
    {
        return response()->json([
            'module' => 'AccessControl',
            'status' => 'running'
        ]);
    }

    /**
     * List roles (placeholder)
     */
    public function roles()
    {
        return response()->json([
            'message' => 'Role list endpoint placeholder'
        ]);
    }

    /**
     * List permissions (placeholder)
     */
    public function permissions()
    {
        return response()->json([
            'message' => 'Permission list endpoint placeholder'
        ]);
    }

    /**
     * Fetch all users
     */
    public function index(Request $request)
    {
        $users = User::with(['company', 'branch', 'team'])->get();
        return response()->json($users);
    }

    /**
     * Get a single user
     */
    public function show($id)
    {
        $user = User::with(['company', 'branch', 'team'])->findOrFail($id);
        return response()->json($user);
    }

    /**
     * Create a new user
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'username' => 'nullable|string|max:255|unique:users,username',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6',
            'company_id' => 'required|exists:companies,id',
            'branch_id' => 'required|exists:branches,id',
            'team_id' => 'nullable|exists:teams,id',
            'is_active' => 'boolean',
            'phone' => 'nullable|string',
            'mobile' => 'nullable|string',
            'address_line1' => 'nullable|string',
            'address_line2' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'postal_code' => 'nullable|string',
            'country' => 'nullable|string',
            'job_title' => 'nullable|string',
            'timezone' => 'nullable|string|max:100',
            'language' => 'nullable|string|max:5',
        ]);

        $data['password'] = Hash::make($data['password']);

        $user = User::create($data);
        return response()->json($user, 201);
    }

    /**
     * Update an existing user
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $data = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'username' => [
                'sometimes',
                'nullable',
                'string',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
            'email' => [
                'sometimes',
                'required',
                'email',
                Rule::unique('users')->ignore($user->id),
            ],
            'password' => 'sometimes|nullable|string|min:6',
            'company_id' => 'sometimes|required|exists:companies,id',
            'branch_id' => 'sometimes|required|exists:branches,id',
            'team_id' => 'sometimes|nullable|exists:teams,id',
            'is_active' => 'sometimes|boolean',
            'phone' => 'nullable|string',
            'mobile' => 'nullable|string',
            'address_line1' => 'nullable|string',
            'address_line2' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'postal_code' => 'nullable|string',
            'country' => 'nullable|string',
            'job_title' => 'nullable|string',
            'timezone' => 'nullable|string|max:100',
            'language' => 'nullable|string|max:5',
        ]);

        if (isset($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }

        $user->update($data);
        return response()->json($user);
    }

    /**
     * Delete a user
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();
        return response()->json(['message' => 'User deleted successfully']);
    }
}
