<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        $prefix = 'user_';

        // ----------------------
        // Roles
        // ----------------------
        Schema::create($prefix.'roles', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->unsignedBigInteger('company_id')->nullable(); // optional
            $table->timestampsTz();
        });

        // ----------------------
        // Permissions
        // ----------------------
        Schema::create($prefix.'permissions', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('module')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Role Permissions
        // ----------------------
        Schema::create($prefix.'role_permissions', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('role_id');       // removed constrained
            $table->unsignedBigInteger('permission_id'); // removed constrained
            $table->timestampsTz();
            $table->unique(['role_id','permission_id']);
        });

        // ----------------------
        // Teams
        // ----------------------
        Schema::create($prefix.'teams', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id')->nullable();
            $table->string('name');
            $table->timestampsTz();
        });

        // ----------------------
        // Users
        // ----------------------
        Schema::create('users', function (Blueprint $table) use ($prefix) {
            $table->id();
            
            $table->string('name');
            $table->string('username')->nullable()->unique();
            $table->string('email')->unique();
            $table->timestamp('email_verified_at')->nullable();
            $table->string('password');
            $table->unsignedBigInteger('team_id')->nullable(); // removed constrained
            $table->boolean('is_active')->default(true);
            $table->integer('failed_logins')->default(0);
            $table->timestamp('locked_until')->nullable();
            $table->string('two_factor_secret')->nullable();
            $table->text('two_factor_recovery_codes')->nullable();
            $table->string('avatar')->nullable();
            $table->string('job_title')->nullable();
            $table->string('timezone', 100)->default('UTC');
            $table->string('language', 5)->default('en');

            // Contact fields
            $table->string('phone')->nullable();
            $table->string('mobile')->nullable();

            // Address fields
            $table->string('address_line1')->nullable();
            $table->string('address_line2')->nullable();
            $table->string('city')->nullable();
            $table->string('state')->nullable();
            $table->string('postal_code')->nullable();
            $table->string('country', 100)->nullable();

            $table->rememberToken();
            $table->timestampsTz();
            $table->softDeletesTz();
        });

        // ----------------------
        // User ↔ Company & Branch Pivot
        // ----------------------
        Schema::create('user_companies', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');   // removed constrained
            $table->unsignedBigInteger('company_id'); 
            $table->unsignedBigInteger('branch_id')->nullable();
            $table->enum('role_type', ['employee','customer','supplier','partner'])->default('employee');
            $table->timestampsTz();

            $table->unique(['user_id','company_id','branch_id']);
            $table->index(['user_id','company_id','branch_id']);
        });

        // ----------------------
        // Dynamic Attributes
        // ----------------------
        Schema::create($prefix.'attributes', function (Blueprint $table) {
            $table->id();
            
            $table->unsignedBigInteger('user_id'); // removed constrained
            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('branch_id')->nullable();
            $table->string('attribute_name');
            $table->string('value')->nullable();
            $table->enum('attribute_type', ['string','number','date','select'])->default('string');
            $table->boolean('is_required')->default(false);
            $table->timestampsTz();
        });

        // ----------------------
        // Login History
        // ----------------------
        Schema::create($prefix.'logins', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id'); // removed constrained
            $table->string('ip_address')->nullable();
            $table->string('user_agent')->nullable();
            $table->timestamp('login_at')->useCurrent();
        });

        // ----------------------
        // Password Reset Tokens
        // ----------------------
        Schema::create('password_reset_tokens', function (Blueprint $table) {
            $table->string('email')->primary();
            $table->string('token');
            $table->timestamp('created_at')->nullable();
        });

        // ----------------------
        // Personal Access Tokens (Sanctum)
        // ----------------------
        Schema::create('personal_access_tokens', function (Blueprint $table) {
            $table->id();
            $table->morphs('tokenable');
            $table->string('name');
            $table->string('token', 64)->unique();
            $table->text('abilities')->nullable();
            $table->timestamp('last_used_at')->nullable();
            $table->timestamp('expires_at')->nullable();
            $table->timestampsTz();
            $table->softDeletesTz();
        });
    }

    public function down(): void
    {
        $prefix = 'user_';

        Schema::dropIfExists('personal_access_tokens');
        Schema::dropIfExists($prefix.'logins');
        Schema::dropIfExists($prefix.'attributes');
        Schema::dropIfExists('user_companies');
        Schema::dropIfExists('users');
        Schema::dropIfExists($prefix.'teams');
        Schema::dropIfExists($prefix.'role_permissions');
        Schema::dropIfExists($prefix.'permissions');
        Schema::dropIfExists($prefix.'roles');
        Schema::dropIfExists('password_reset_tokens');
    }
};
