<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // ----------------------
        // Departments (optional)
        // ----------------------
        Schema::create('hrm_departments', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('description')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Designations (optional)
        // ----------------------
        Schema::create('hrm_designations', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('description')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Employees
        // ----------------------
        Schema::create('hrm_employees', function (Blueprint $table) {
            $table->id();

            // Removed FKs
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('branch_id')->nullable();
            $table->unsignedBigInteger('team_id')->nullable();
            $table->unsignedBigInteger('department_id')->nullable();
            $table->unsignedBigInteger('designation_id')->nullable();

            $table->string('employee_code')->nullable()->unique();
            $table->date('joining_date')->nullable();
            $table->date('date_of_birth')->nullable();
            $table->enum('gender', ['male','female','other'])->nullable();
            $table->string('marital_status', 20)->nullable();
            $table->string('blood_group', 5)->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
            $table->softDeletesTz();
        });

        // ----------------------
        // Attendance
        // ----------------------
        Schema::create('hrm_attendance', function (Blueprint $table) {
            $table->id();

            // Removed FK
            $table->unsignedBigInteger('employee_id');

            $table->date('attendance_date');
            $table->timestamp('check_in_at')->nullable();
            $table->timestamp('check_out_at')->nullable();
            $table->string('check_in_location')->nullable();
            $table->string('check_out_location')->nullable();
            $table->enum('status', ['present','absent','half_day','leave'])->default('present');
            $table->decimal('hours_worked', 5, 2)->nullable();
            $table->decimal('overtime_hours', 5, 2)->nullable();
            $table->timestampsTz();
            $table->unique(['employee_id','attendance_date']);
        });

        // ----------------------
        // Leave Requests
        // ----------------------
        Schema::create('hrm_leaves', function (Blueprint $table) {
            $table->id();

            // Removed FKs
            $table->unsignedBigInteger('employee_id');
            $table->unsignedBigInteger('approved_by')->nullable();

            $table->enum('leave_type', ['casual','sick','annual','maternity','paternity','unpaid','half_day']);
            $table->date('start_date');
            $table->date('end_date');
            $table->text('reason')->nullable();
            $table->enum('status', ['pending','approved','rejected'])->default('pending');
            $table->timestampsTz();
        });

        // ----------------------
        // Performance & Appraisals
        // ----------------------
        Schema::create('hrm_performance', function (Blueprint $table) {
            $table->id();

            // Removed FKs
            $table->unsignedBigInteger('employee_id');
            $table->unsignedBigInteger('reviewed_by')->nullable();

            $table->text('objectives')->nullable();
            $table->text('achievements')->nullable();
            $table->integer('rating')->nullable();
            $table->text('feedback')->nullable();
            $table->date('review_date')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Employee Documents
        // ----------------------
        Schema::create('hrm_employee_documents', function (Blueprint $table) {
            $table->id();

            // Removed FK
            $table->unsignedBigInteger('employee_id');

            $table->string('document_name');
            $table->string('file_path');
            $table->string('document_type')->nullable();
            $table->timestampsTz();
        });

        // ----------------------
        // Employee Skills / Certifications (optional)
        // ----------------------
        Schema::create('hrm_employee_skills', function (Blueprint $table) {
            $table->id();

            // Removed FK
            $table->unsignedBigInteger('employee_id');

            $table->string('skill_name');
            $table->string('certification')->nullable();
            $table->date('certified_at')->nullable();
            $table->timestampsTz();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('hrm_employee_skills');
        Schema::dropIfExists('hrm_employee_documents');
        Schema::dropIfExists('hrm_performance');
        Schema::dropIfExists('hrm_leaves');
        Schema::dropIfExists('hrm_attendance');
        Schema::dropIfExists('hrm_employees');
        Schema::dropIfExists('hrm_designations');
        Schema::dropIfExists('hrm_departments');
    }
};
