<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {

    public function up(): void
    {
        /*-------------------------------------------------------
         | 1️⃣ Tailor Staff Types (Users Extended)
         -------------------------------------------------------*/
        Schema::create('tailor_staff', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id'); // removed FK

            $table->enum('role', ['tailor', 'cutter', 'shalenger', 'ironman', 'qc']);
            $table->string('specialization')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 2️⃣ Measurement Templates
         -------------------------------------------------------*/
        Schema::create('measurement_templates', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('company_id'); // removed FK

            $table->string('name');
            $table->boolean('is_default')->default(false);

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 3️⃣ Measurement Template Fields
         -------------------------------------------------------*/
        Schema::create('measurement_template_fields', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('template_id'); // removed FK

            $table->string('name');
            $table->enum('input_type', ['text', 'number', 'select', 'radio']);
            $table->json('options')->nullable();
            $table->string('unit')->nullable();

            $table->boolean('is_required')->default(false);
            $table->integer('sort_order')->default(0);

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 4️⃣ Customer Measurements (VERSIONED)
         -------------------------------------------------------*/
        Schema::create('customer_measurements', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('customer_id'); // removed FK

            $table->unsignedBigInteger('template_id'); // removed FK

            $table->unsignedInteger('version')->default(1);

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 5️⃣ Customer Measurement Values
         -------------------------------------------------------*/
        Schema::create('customer_measurement_values', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('measurement_id'); // removed FK
            $table->unsignedBigInteger('field_id'); // removed FK

            $table->string('value')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 6️⃣ Tailor Orders (MAIN)
         -------------------------------------------------------*/
        Schema::create('tailor_orders', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('company_id'); // removed FK

            $table->unsignedBigInteger('customer_id'); // removed FK

            $table->unsignedBigInteger('template_id'); // removed FK

            $table->string('order_no')->unique();
            $table->enum('priority', ['normal','urgent','vip'])->default('normal');

            $table->decimal('amount', 10, 2)->default(0);
            $table->decimal('advance', 10, 2)->default(0);
            $table->decimal('remaining', 10, 2)->default(0);

            $table->enum('status', [
                'pending_cutting',
                'cutting',
                'pending_stitching',
                'stitching',
                'pending_finishing',
                'finishing',
                'qc_pending',
                'qc_passed',
                'completed',
                'delivered'
            ])->default('pending_cutting');

            $table->date('delivery_date')->nullable();

            $table->string('barcode')->unique()->nullable();
            $table->string('image')->nullable();

            $table->timestampsTz();
            $table->softDeletesTz();
        });

        /*-------------------------------------------------------
         | 7️⃣ Order Notes
         -------------------------------------------------------*/
        Schema::create('tailor_order_notes', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('order_id'); // removed FK
            $table->unsignedBigInteger('user_id')->nullable(); // removed FK

            $table->text('note');
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 8️⃣ Order Assignment Workflow
         -------------------------------------------------------*/
        Schema::create('tailor_order_assignments', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('order_id'); // removed FK
            $table->unsignedBigInteger('staff_id'); // removed FK

            $table->enum('task', ['cutting', 'stitching', 'finishing', 'qc']);
            $table->enum('status', ['pending', 'in_progress', 'completed'])->default('pending');

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 9️⃣ Storage / Shelf Locations
         -------------------------------------------------------*/
        Schema::create('tailor_shelves', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('company_id'); // removed FK

            $table->string('rack');
            $table->string('shelf');
            $table->string('section')->nullable();

            $table->timestampsTz();
        });

        Schema::create('tailor_order_storage', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('order_id'); // removed FK
            $table->unsignedBigInteger('shelf_id'); // removed FK

            $table->string('position')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 🔟 QC Logs
         -------------------------------------------------------*/
        Schema::create('tailor_qc_logs', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('order_id'); // removed FK
            $table->unsignedBigInteger('staff_id')->nullable(); // removed FK

            $table->enum('result', ['pass','fail']);
            $table->text('remarks')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 1️⃣1️⃣ Notifications
         -------------------------------------------------------*/
        Schema::create('tailor_notifications', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('order_id')->nullable(); // removed FK
            $table->unsignedBigInteger('customer_id')->nullable(); // removed FK

            $table->enum('type', ['sms','email','whatsapp']);
            $table->string('to');
            $table->text('message');
            $table->boolean('is_sent')->default(false);

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
         | 1️⃣2️⃣ Batch Processing
         -------------------------------------------------------*/
        Schema::create('tailor_batches', function (Blueprint $table) {
            $table->id();

            $table->string('batch_no')->unique();
            $table->unsignedBigInteger('staff_id')->nullable(); // removed FK

            $table->enum('task', ['cutting','stitching','finishing'])->default('cutting');

            $table->timestampsTz();
        });

        Schema::create('tailor_batch_items', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('batch_id'); // removed FK
            $table->unsignedBigInteger('order_id'); // removed FK

            $table->timestampsTz();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('tailor_batch_items');
        Schema::dropIfExists('tailor_batches');
        Schema::dropIfExists('tailor_notifications');
        Schema::dropIfExists('tailor_qc_logs');
        Schema::dropIfExists('tailor_order_storage');
        Schema::dropIfExists('tailor_shelves');
        Schema::dropIfExists('tailor_order_assignments');
        Schema::dropIfExists('tailor_order_notes');
        Schema::dropIfExists('tailor_orders');
        Schema::dropIfExists('customer_measurement_values');
        Schema::dropIfExists('customer_measurements');
        Schema::dropIfExists('measurement_template_fields');
        Schema::dropIfExists('measurement_templates');
        Schema::dropIfExists('tailor_staff');
    }
};
